# encoding: utf-8
"""
OOS Management Python API Example
"""
from ctyun.config import Config
from ctyun.auth.credentials import Credentials
import ctyun.exceptions as exceptions
from ctyun.man.client import ManagementClient
from ctyun.man.model.getAvailBw import GetAvailBwRequest
from ctyun.man.model.getAvailableBandwidth import GetAvailableBandwidthRequest
from ctyun.man.model.getBandwidth import GetBandwidthRequest
from ctyun.man.model.getCapacity import GetCapacityRequest
from ctyun.man.model.getConnection import GetConnectionRequest
from ctyun.man.model.getUsage import GetUsageRequest
from ctyun.man.model.getDeleteCapacity import GetDeleteCapacityRequest
from ctyun.man.model.getTraffics import GetTrafficsRequest
from ctyun.man.model.getRequests import GetRequestsRequest
from ctyun.man.model.getReturnCode import GetReturnCodeRequest
from ctyun.man.model.getConcurrentConnection import GetConcurrentConnectionRequest
import traceback

ACCESS_KEY = 'your_accessKey'
SECRET_KEY = 'your_secretKey'
MANAGEMENT_ENDPOINT = 'your_domain'
SIGNATURE_VERSION = 'v4'  # Algorithm version used for calculating signatures, v2 version:'v2', v4 version:'v4'
BUCKET = 'your_bucket'  # bucket name

# Preprocessing, setting up access to Statistics services
if MANAGEMENT_ENDPOINT.lower().find("http") < 0 and MANAGEMENT_ENDPOINT.lower().find("https") < 0:
    MANAGEMENT_ENDPOINT = "http://" + MANAGEMENT_ENDPOINT
try:
    config = Config()
    config.signature_version = SIGNATURE_VERSION
    credentials = Credentials(ACCESS_KEY, SECRET_KEY)
    client = ManagementClient(credentials, config, MANAGEMENT_ENDPOINT, False)
except Exception as ex:
    print(traceback.format_exc())
    print(ex)
    exit(-1)


def handle_error(fn):
    """
    Exception handling
    """

    def inner():
        try:
            fn()
        except exceptions.ServiceException as e:
            print(e.get_header().to_string())
            print(e.to_string())

    return inner


# 1	GetCapacity
@handle_error
def get_capacity_example():
    """
    This operation is used to query user's capacity
    """
    request = GetCapacityRequest()
    # Specify the start time for querying capacity
    request.BeginDate = "2024-08-20"
    # Specify the end time for querying capacity
    request.EndDate = "2024-08-21"
    # Name of the bucket
    request.Bucket = BUCKET
    # Specify the sampling granularity for returning statistics
    request.Freq = "byDay"
    # Specify the storage class
    request.StorageClass = "STANDARD"
    # Specify the data location for querying capacity.
    # request.Region = "QingDao"

    result = client.get_capacity(request)

    print(result.get_header().to_string())
    print(result.to_string())


# 2	GetDeleteCapacity
@handle_error
def get_delete_capacity_example():
    """
    This operation is used to query the capacity deleted by the user
    """
    request = GetDeleteCapacityRequest()
    # Specify the start time for querying capacity
    request.BeginDate = "2024-08-20"
    # Specify the end time for querying capacity
    request.EndDate = "2024-08-21"
    # Name of the bucket
    request.Bucket = BUCKET
    # Specify the sampling granularity for returning statistics
    request.Freq = "byDay"
    # Specify the storage class
    request.StorageClass = "STANDARD"
    # Specify the data location for deleting capacity.
    # request.Region = "QingDao"

    result = client.get_delete_capacity(request)

    print(result.get_header().to_string())
    print(result.to_string())


# 3 GetTraffics
@handle_error
def get_traffics_example():
    """
    This operation is used to query user's traffic
    """
    request = GetTrafficsRequest()
    # Specify the start time for querying traffic
    request.BeginDate = "2024-08-20"
    # Specify the end time for querying traffic
    request.EndDate = "2024-08-21"
    # Name of the bucket
    request.Bucket = BUCKET
    # Specify the sampling granularity for returning statistics
    request.Freq = "byDay"
    # Specify the returned type of traffic, values: all/inbound/outbound
    request.InOutType = "all"
    # Specify the returned type of traffic, values: all/internet/noninternet
    request.InternetType = "all"
    # Specify the returned type of traffic, values: all/direct/roam
    request.TrafficsType = "all"
    # Specify the storage class
    request.StorageClass = "STANDARD"
    # Specify the data location for querying traffic.
    # request.Region = "QingDao"

    result = client.get_traffics(request)

    print(result.get_header().to_string())
    print(result.to_string())


# 4 GetRequests
@handle_error
def get_requests_example():
    """
    This operation is used to query the number of user requests
    """
    request = GetRequestsRequest()
    # Specify the start time for querying the requests
    request.BeginDate = "2024-08-20"
    # Specify the end time for querying the request
    request.EndDate = "2024-08-21"
    # Name of the bucket
    request.Bucket = BUCKET
    # Specify the sampling granularity for returning statistics
    request.Freq = "byDay"
    # Specify the returned requests, values: all/internet/noninternet
    request.InternetType = "all"
    # Specify the returned type of request, values: all/get/head/put/post/delete/others
    request.RequestsType = "all"
    # Specify the storage class
    request.StorageClass = "STANDARD"
    # Specify the data location for querying the requests.
    # request.Region = "QingDao"

    result = client.get_requests(request)

    print(result.get_header().to_string())
    print(result.to_string())


# 5 GetReturnCode
@handle_error
def get_return_code_example():
    """
    This operation is used to query the number of user request return codes
    """
    request = GetReturnCodeRequest()
    # Specify the start time for querying the return codes
    request.BeginDate = "2024-08-20"
    # Specify the end time for querying the return codes
    request.EndDate = "2024-08-21"
    # Name of the bucket
    request.Bucket = BUCKET
    # Specify the sampling granularity for returning statistics
    request.Freq = "byDay"
    # Specify the network type of return code, values: all/internet/noninternet
    request.InternetType = "all"
    # Specify the request type of return code, values: all/get/head/put/post/delete/others
    request.RequestsType = "all"
    # Specify the returned type of return code
    request.ResponseType = "all"
    # Specify the storage class
    request.StorageClass = "STANDARD"
    # Specify the data location for querying the return code.
    # request.Region = "QingDao"

    result = client.get_return_code(request)

    print(result.get_header().to_string())
    print(result.to_string())


# 6 GetConcurrentConnection
@handle_error
def get_concurrent_connection_example():
    """
    This operation is used to query the number of user's concurrent connections
    """
    request = GetConcurrentConnectionRequest()
    # Specify the start time for querying the number of concurrent connections
    request.BeginDate = "2024-07-20"
    # Specify the end time for querying the number of concurrent connections
    request.EndDate = "2024-07-20"
    # Name of the bucket
    request.Bucket = BUCKET
    # Specify the sampling granularity for returning statistics
    request.Freq = "by5min"
    # Specify the network type of return code, values: all/internet/noninternet
    request.InternetType = "all"
    # Specify the data location for querying the number of concurrent connections.
    # request.Region = "QingDao"

    result = client.get_concurrent_connection(request)

    print(result.get_header().to_string())
    print(result.to_string())


# 7	GetUsage
@handle_error
def get_usage_example():
    """
    This operation is used to query the usage of user's bucket
    """
    request = GetUsageRequest()
    # Specify the start time of the query
    request.BeginDate = "2024-08-20"
    # Specify the end time of the query
    request.EndDate = "2024-08-21"
    # Name of the bucket
    request.Bucket = BUCKET

    result = client.get_usage(request)

    print(result.get_header().to_string())
    print(result.to_string())


# 8	GetBandwidth
@handle_error
def get_bandwidth_example():
    """
    This operation is used to query the user's used bandwidth
    """
    request = GetBandwidthRequest()
    # Specify the start time for querying the used bandwidth
    request.BeginDate = "2024-08-20-00-00"
    # Specify the end time for querying the used bandwidth
    request.EndDate = "2024-08-21-10-10"
    # Name of the bucket
    request.Bucket = BUCKET

    result = client.get_bandwidth(request)

    print(result.get_header().to_string())
    print(result.to_string())


# 9 GetAvailableBandwidth
@handle_error
def get_available_bandwidth_example():
    """
    This operation is used to query the user's available bandwidth
    """
    request = GetAvailableBandwidthRequest()
    # Specify the start time for querying the available bandwidth
    request.BeginDate = "2024-08-20"
    # Specify the end time for querying the available bandwidth
    request.EndDate = "2024-08-20"
    # Specify the returned type of traffic
    request.InOutType = "all"
    # Specify the returned type of traffic
    request.InternetType = "all"
    # Specify the sampling granularity for returning statistics
    request.Freq = "by5min"

    result = client.get_available_bandwidth(request)

    print(result.get_header().to_string())
    print(result.to_string())


# 10 GetAvailBW
@handle_error
def get_avail_bw_example():
    """
    This operation is used to obtain the bandwidth usage
    """
    request = GetAvailBwRequest()
    # Specify the start time for querying the bandwidth usage
    request.BeginDate = "2024-08-20-00-05"
    # Specify the end time for querying the bandwidth usage
    request.EndDate = "2024-08-20-15-00"

    result = client.get_avail_bw(request)

    print(result.get_header().to_string())
    print(result.to_string())


# 11 GetConnection
@handle_error
def get_connection_example():
    """
    This operation is used to obtain the status of concurrent connection
    """
    request = GetConnectionRequest()
    # Specify the start time for querying the status of concurrent connection
    request.BeginDate = "2024-08-20-00-00"
    # Specify the end time for querying the status of concurrent connection
    request.EndDate = "2024-08-20-00-05"
    # Name of the bucket
    request.Bucket = BUCKET

    result = client.get_connection(request)

    print(result.get_header().to_string())
    print(result.to_string())


if __name__ == '__main__':
    try:
        # 1 GetCapacity
        get_capacity_example()

        # 2 GetDeleteCapacity
        # get_delete_capacity_example()

        # 3 GetTraffics
        # get_traffics_example()

        # 4 GetRequests
        # get_requests_example()

        # 5 GetReturnCode
        # get_return_code_example()

        # 6 GetConcurrentConnection
        # get_concurrent_connection_example()

        # 7 get_usage
        # get_usage_example()

        # 8 get_bandwidth
        # get_bandwidth_example()

        # 9 GetAvailableBandwidth
        # get_available_bandwidth_example()

        # 10 GetAvailBW
        # get_avail_bw_example()

        # 11 GetConnection
        # get_connection_example()

    except Exception as ex:
        print(traceback.format_exc())
        print(ex)
